/***************************************************************
 *                   Materials Object Library                  *
 * Class transversely_isotropic_elastic_tensor : for           *
 *  transversely isotropic elastic materials (in x3 direction) *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2008,2009 BERBENNI Stephane
 * copyright © 2009,2011,2012 COLLARD Christophe
 * copyright © 2008,2009,2011,2012 Centre National de la Recherche Scientifique
 * copyright © 2008,2009,2011 Arts et Métiers ParisTech
 * copyright © 2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2011 Laboratoire d'Etude des Microstructures et de Mécanique des Matériaux (LEM3 - CNRS)
 * copyright © 2011,2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/


/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::transversely_isotropic_elastic_tensor
    \brief compuation for \f$ 4^{th} \f$ order transversely isotropic elastic tensor

    \htmlonly 
    <FONT color="#838383">

    transversely isotropic elastic tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    The stress tensor \f$ \boldsymbol\sigma \f$ is related to the strain tensor \f$ \boldsymbol{\EuScript{E}} \f$ through the \f$ 4^\text{th} \f$ order elasticity tensor \f$ \textbf{C} \f$ by : \n
    \f$ \sigma_{ij} = C_{ijkl} \EuScript{E}_{kl} \f$. \n
    By taking into account the symmetry of the strain and stress tensors, 
    \f$ \sigma_{ij} = C_{ijkl} \EuScript{E}_{kl} = C_{ijkl} \EuScript{E}_{lk} = \sigma_{ji} \f$, \n
    we get \f$C_{ijkl} = C_{jikl} = C_{ijlk} \f$.\n
    Moreover, if we take into account the symmetry property of the elasticity tensor \f$ C_{ijkl} = C_{klij} \f$, the \f$ 81 \f$ coefficients of the elasticity tensor reduces to 21 coefficients. \n
    Then, the relation between the stress tensor and the strain tensor can be written as : \n
    \f$ \begin{pmatrix} \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \sigma_{12} \\ \sigma_{13} \\ \sigma_{23} \end{pmatrix} =
         \begin{pmatrix} c_{11} & c_{12} & c_{13} & c_{14} & c_{15} & c_{16} \\ 
                                & c_{22} & c_{23} & c_{24} & c_{25} & c_{26} \\
			        &        & c_{33} & c_{34} & c_{35} & c_{36} \\
			        & sym    &        & c_{44} & c_{45} & c_{46} \\
			        &        &        &        & c_{55} & c_{56} \\
			        &        &        &        &        & c_{66} \\
        \end{pmatrix} 
	\begin{pmatrix} \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ 2 \EuScript{E}_{12} \\ 2 \EuScript{E}_{13} \\ 2 \EuScript{E}_{23} \end{pmatrix} \f$ \n
    where \f$ \displaystyle c_{xy} = C_{XY} \f$, with the following relations between the tensor indices and the matrix indices  \n
    \f$ \text{\begin{tabular}{|c|c|c|c|c|c|c|c|c|c|}
    \hline 
    Tensor & 11 & 22 & 33 & 12 & 21 & 13 & 31 & 23 & 32 \\
    \hline
    Matrix & 1 & 2 & 3 & 4 & 4 & 5 & 5 & 6 & 6 \\
    \hline
    \end{tabular}} \f$


    For materials with transversely isotropic symmetry, the non zero elements of the elasticity tensor read \n

    \f$ \displaystyle C_{1111} = C_{2222} = \frac{E_1}{2 (1 + \nu_{12})} + k \f$ \n
    \f$ \displaystyle C_{3333} = 2 \frac{E_3}{E_1} k (1 - \nu_{12}) \f$ \n
    \f$ \displaystyle C_{1122} = C_{2211} =  k - \frac{E_1}{2 (1 + \nu_{12})} \f$ \n
    \f$ \displaystyle C_{1133} = C_{3311} = C_{2233} = C_{3322} = 2  k \nu_{31} \f$ \n
    \f$ \displaystyle C_{1212} = C_{1221} = C_{2112} = C_{2121} = \frac{1}{2} (C_{1111} - C_{1122}) = \frac{E_1}{2(1+\nu_{12})} \f$ \n
    \f$ C_{1313} = C_{1331} = C_{3113} = C_{3131} = C_{2323} = C_{2332} = C_{3223} = C_{3232} = G_{13} \f$ \n
    where \f$ E_1, E_3, \nu_{12}, \nu_{31}, G_{13} \f$ are the transversely isotropic constants, and \f$ k \f$ is the bulk modulus defined by: \n
    \f$ \displaystyle 2 k = \frac{E_1 E_3}{E_3 (1 - \nu_{12}) - 2 E_1 \nu_{31}^2}. \f$ \n

    The associated reduced symmetric matrix reads : \n
    \f$ \begin{pmatrix} c_{11} & c_{12} & c_{13} & 0 & 0 & 0 \\ 
                        c_{21} & c_{22} & c_{23} & 0 & 0 & 0 \\
			c_{31} & c_{32} & c_{33} & 0 & 0 & 0 \\
			0 & 0 & 0 & c_{44} & 0 & 0 \\
			0 & 0 & 0 & 0 & c_{55} & 0 \\
			0 & 0 & 0 & 0 & 0 & c_{66}
        \end{pmatrix} \f$ \n
    with the following relations : \n
    \f$ \displaystyle c_{11} = c_{22} = \frac{E_1}{2 (1 + \nu_{12})} + k \f$ \n
    \f$ \displaystyle c_{33} = 2 \frac{E_3}{E_1} k (1 - \nu_{12}) \f$ \n
    \f$ \displaystyle c_{44} = \frac{1}{2} (C_{11} - C_{12}) = \frac{E_1}{2(1+\nu_{12})} \f$ \n
    \f$ \displaystyle c_{55} = c_{66} =  G_{13} \f$ \n
    \f$ \displaystyle c_{12} = c_{21} =  k - \frac{E_1}{2 (1 + \nu_{12})} \f$ \n
    \f$ c_{13} = c_{31} = c_{23} = c_{32} = 2  k \nu_{31} \f$. \n

    \author copyright \htmlonly &#169; 2008, 2009 St&#233;phane BERBENNI \endhtmlonly \n
            copyright \htmlonly &#169; \endhtmlonly 2009, 2011, 2012 Christophe COLLARD \n
	    copyright \htmlonly &#169; 2008, 2009, 2011, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2008, 2009, 2011 Arts et M&#233;tiers ParisTech \endhtmlonly \n
            copyright \htmlonly &#169; 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2011 Laboratoire d'Etude des Microstructures et de M&#233;canique des Mat&#233;riaux (LEM3 - CNRS) \endhtmlonly \n
            copyright \htmlonly &#169; 2011, 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.4.0
    \date 2008-2019
    \bug none
    \warning none
*/


#ifndef __cplusplus
#error Must use C++ for the type transversely isotropic elastic tensor
#endif

#ifndef __transversely_isotropic_elastic_tensors_hpp
#define __transversely_isotropic_elastic_tensors_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __fstream
#include <fstream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __math_h
#include <math.h>
#endif

#ifndef __string_h
#include <string.h>
#endif

#ifndef __tensors4_hpp
#include "MOL++/tensors4.hpp"
#endif

#ifndef __transversely_isotropic_tensors_hpp
#include "MateriOL++/transversely isotropic tensors.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//======================================================================================================
template <class T> class transversely_isotropic_elastic_tensor : public transversely_isotropic_tensor<T>
//======================================================================================================
{
  using transversely_isotropic_tensor<T>::C11;
  using transversely_isotropic_tensor<T>::C33;
  using transversely_isotropic_tensor<T>::C66;
  using transversely_isotropic_tensor<T>::C12;
  using transversely_isotropic_tensor<T>::C13;
  using transversely_isotropic_tensor<T>::tensor_name;

  protected:
    T E1, E3, nu12, nu31, G13;

  public:
    transversely_isotropic_elastic_tensor (string name = "") : transversely_isotropic_tensor<T> (name) {};  // default constructor
    transversely_isotropic_elastic_tensor (T, T, T, T, T, string = "");
    transversely_isotropic_elastic_tensor (const transversely_isotropic_tensor<T>&); // cast convertion
    ~transversely_isotropic_elastic_tensor () {}  // destructor

    template <class Tf> friend bool operator == (const transversely_isotropic_elastic_tensor<Tf>&, const transversely_isotropic_elastic_tensor<Tf>&);
    template <class Tf> friend bool operator != (const transversely_isotropic_elastic_tensor<Tf>&, const transversely_isotropic_elastic_tensor<Tf>&);
    template <class Tf> friend ostream& operator << (ostream&, transversely_isotropic_elastic_tensor<Tf>&);
};


//=====Public methods for transversely isotropic elastic tensor=========================


/*!
  \brief Constructor for transversely isotropic elastic tensor with the elastic constants

  See formula in the detailed description.

  \param e1 \f$ E_1 \f$ constant
  \param e3 \f$ E_3 \f$ constant
  \param nu_12 \f$ \nu_{12} \f$ constant
  \param nu_31 \f$ \nu_{31} \f$ constant
  \param g13 \f$ G_{13} \f$ constant
  \param name tensor name
*/

//--------------------------------------------------------------------------------------------------------------------------------
template <class T>
transversely_isotropic_elastic_tensor<T>::transversely_isotropic_elastic_tensor (T e1, T e3, T nu_12, T nu_31, T g13, string name)
//--------------------------------------------------------------------------------------------------------------------------------
{
  E1 = e1;
  E3 = e3;
  nu12 = nu_12;
  nu31 = nu_31;
  G13 = g13;

  T coef = E3 * (1 - nu12) - 2 * E1 * power (nu31,2);
  T k = 0.5 * E1 * E3 / coef; // bulk modulus

  C11 = .5 * E1 / (1 + nu12) + k;
  C66 = G13;
  C13 = 2 * k * nu31;
  C12 = k - 0.5 * E1 / (1 + nu12);
  C33 = 2 * E3 / E1 * k * (1 - nu12);

  tensor_name = name;
  transversely_isotropic_tensor<T>::build_tensor();
}


/*!
  \brief Constructor for cast conversion.

  Converts a \f$ \displaystyle 4^{th} \f$ order transversely isotropic tensor into a transversely isotropic elastic tensor.

  \param tensor \f$ \displaystyle 4^{th} \f$ order transversely isotropic tensor
*/

//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------
template <class T>
transversely_isotropic_elastic_tensor<T>::transversely_isotropic_elastic_tensor (const transversely_isotropic_tensor<T>& tensor) : transversely_isotropic_tensor<T> (tensor)
//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------
{
  T c0 = pow (C11, 2) - pow (C12, 2);
  T c1 = C11 * C33  -  pow (C13, 2);
  E1 = E3 = (C33 * c0 +  2 * pow (C13, 2) * (C12 - C11));
  E1 /= c1;
  E3 /= c0;
  nu12 = (C12 * C33 - pow (C13, 2)) / c1;
  nu31 = (C11 * C13 - C12 * C13) / c0;
  G13 = C66;
}


//---------------------------------------------------------------------------------------------------------------------------------------------
template <class Tf> bool operator == (const transversely_isotropic_elastic_tensor<Tf>& t1, const transversely_isotropic_elastic_tensor<Tf>& t2)
//---------------------------------------------------------------------------------------------------------------------------------------------
{
  bool err = relative_error (t1.E1, t2.E1);
  err *= relative_error (t1.E3, t2.E3);
  err *= relative_error (t1.nu12, t2.nu12);
  err *= relative_error (t1.nu31, t2.nu31);
  err *= relative_error (t1.G13, t2.G13);

  return err;
}


//---------------------------------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, transversely_isotropic_elastic_tensor<Tf>& tsr)
//---------------------------------------------------------------------------------------------------
{
  s << tsr.tensor_name << endl;
  s << "Material constants : (" << tsr.E1 << ", " << tsr.E3 << ", " << tsr.nu12 << ", " << tsr.nu31 << ", " << tsr.G13 << ")" << endl;
  s << tsr.mat;
  s << endl;

  return s;
}


}


#endif
